package gov.va.genisis2.bo;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Properties;
import java.util.UUID;

import org.apache.commons.lang3.StringUtils;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import gov.va.genisis2.common.enums.CommonEnum;
import gov.va.genisis2.common.enums.RoleTypeEnum;
import gov.va.genisis2.common.enums.WorkflowStatusEnum;
import gov.va.genisis2.configuration.GenisisPropertiesUtil;
import gov.va.genisis2.converter.CommentHistoryConverter;
import gov.va.genisis2.converter.RequestConverter;
import gov.va.genisis2.converter.RequestHistoryConverter;
import gov.va.genisis2.converter.RequestTypeConverter;
import gov.va.genisis2.converter.SourceConverter;
import gov.va.genisis2.converter.StudyApprovalConverter;
import gov.va.genisis2.converter.UserConverter;
import gov.va.genisis2.converter.UserCountConverter;
import gov.va.genisis2.converter.UserRefreshHistoryConverter;
import gov.va.genisis2.converter.WorkflowStatusConverter;
import gov.va.genisis2.dto.CommentHistoryDTO;
import gov.va.genisis2.dto.RequestDTO;
import gov.va.genisis2.dto.RequestHistoryDTO;
import gov.va.genisis2.dto.StudyApprovalDTO;
import gov.va.genisis2.dto.UserDTO;
import gov.va.genisis2.exceptions.ErrorEnum;
import gov.va.genisis2.exceptions.GenisisServiceException;
import gov.va.genisis2.model.CommentHistory;
import gov.va.genisis2.model.CommentType;
import gov.va.genisis2.model.Request;
import gov.va.genisis2.model.RequestHistory;
import gov.va.genisis2.model.RequestType;
import gov.va.genisis2.model.Source;
import gov.va.genisis2.model.StudyApproval;
import gov.va.genisis2.model.User;
import gov.va.genisis2.model.UserCount;
import gov.va.genisis2.model.UserRefreshHistory;
import gov.va.genisis2.model.WorkflowStatus;
import gov.va.genisis2.service.impl.CommentHistoryService;
import gov.va.genisis2.service.impl.LookUpService;
import gov.va.genisis2.service.impl.RequestService;
import gov.va.genisis2.service.impl.StudyApprovalService;
import gov.va.genisis2.service.impl.UserManagementService;
import gov.va.genisis2.util.EhcacheHelper;
import gov.va.genisis2.util.rest.helper.ResponseWrapper;
import gov.va.genisis2.vo.CopyTableDomains;
import gov.va.genisis2.vo.CopyTableSource;

/**
 * The Class BusinessService.
 * 
 * The BusinessService Class defines the methods that are called by the
 * Genisis2Controller to perform the business logic.
 * 
 */
/**
 * @author Paramjeet Bindra
 *
 */
@Service
public class BusinessService implements IBusiness {
	/** The LOGGER. */
	private static final Logger LOGGER = LoggerFactory.getLogger(BusinessService.class);
	
	/** The request service. */
	private RequestService requestService;
	
	/** The study approval service. */
	private StudyApprovalService studyApprovalService;
	
	/** The comment history service. */
	private CommentHistoryService commentHistoryService;
	
	/** The user management service. */
	private UserManagementService userManagementService;
	
	/** The activiti service. */
	private ActivitiService activitiService;
	
	/** The look up service. */
	private LookUpService lookUpService;
	
	/** The copy table source. */
	private CopyTableSource copyTableSource;
	
	/** The copy table domain details */
	private CopyTableDomains copyTableDomains;
	
	@Autowired
	private GenisisPropertiesUtil genisisPropertiesUtil;
	
	@Autowired
	private RequestConverter requestConverter;
	
	@Autowired
	private UserConverter userConverter;
	
	@Autowired
	private StudyApprovalConverter studyApprovalConverter;
	
	@Autowired
	private CommentHistoryConverter commentHistoryConverter;
	
	@Autowired
	private SourceConverter sourceConverter;
	
	@Autowired
	private RequestTypeConverter requestTypeConverter;
	
	@Autowired
	private RequestHistoryConverter requestHistoryConverter;
	
	@Autowired
	private UserCountConverter userCountConverter;
	
	@Autowired
	private UserRefreshHistoryConverter userRefreshHistoryConverter;
	
	@Autowired
	private WorkflowStatusConverter workflowStatusConverter;
	
	@Autowired
	private BusinessHelper businessHelper;
	
	@Autowired
	private EmailHelper emailHelper;
	
	@Autowired
	private EhcacheHelper ehcacheHelper;

	/**
	 * Gets Genisis2 API version from pom.properties
	 * 
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getGenisisAPIDetails() throws GenisisServiceException {
		Properties properties = new Properties();
		ResponseWrapper wrapper = new ResponseWrapper();
		InputStream inputStream;
		String apiVersion;
		try {
			inputStream = getClass().getResourceAsStream(genisisPropertiesUtil.getGenisisPomPropsPath());
			if (inputStream != null) {
				properties.load(inputStream);
			}
			apiVersion = properties.getProperty(CommonEnum.VERSION.getText(), null);
			wrapper.setResponse(apiVersion);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
		} catch (IOException ex) {
			LOGGER.error("Exception occured on createStudyApproval.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		return wrapper;
	}

	/**
	 * Creates the study approval.
	 *
	 * @param studyApprovalDto
	 *            The studyApprovalDto.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper createStudyApproval(StudyApprovalDTO studyApprovalDto) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  createStudyApproval");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval;
		if (null == studyApprovalDto) {
			wrapper.setResponse(studyApprovalDto);
			wrapper.setSuccess(false);
			wrapper.setMessage("Study Approval Request Payload is null");
		} else {
			studyApproval = studyApprovalConverter.convert(studyApprovalDto);
			int id = 0;
			try {
				id = studyApprovalService.createStudyApproval(studyApproval);
			} catch (Exception ex) {
				LOGGER.error("Exception occurred on  createStudyApproval.", ex);
				throw new GenisisServiceException(ex.getMessage(), ex);
			}
			if (id > 0) {
				studyApprovalDto.setId(id);
				wrapper.setResponse(studyApprovalDto);
				wrapper.setSuccess(true);
				wrapper.setMessage(null);
			} else {
				wrapper.setResponse(studyApprovalDto);
				wrapper.setSuccess(false);
				wrapper.setMessage("Unable to create Study Approval");
			}
		}
		return wrapper;
	}

	/**
	 * Update study approval.
	 *
	 * @param studyApprovalDto
	 *            The studyApprovalDto.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper updateStudyApproval(StudyApprovalDTO studyApprovalDto) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: updateStudyApproval");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApproval studyApproval;
		if (null == studyApprovalDto) {
			wrapper.setResponse(studyApprovalDto);
			wrapper.setSuccess(false);
			wrapper.setMessage("Study Approval Request Payload is null");
		} else {
			int id = studyApprovalDto.getId();
			if (id > 0) {
				studyApproval = studyApprovalConverter.convert(studyApprovalDto);
				try {
					studyApprovalService.updateStudyApproval(studyApproval);
				} catch (Exception ex) {
					LOGGER.error("Exception occurred on  updateStudyApproval.", ex);
					throw new GenisisServiceException(ex.getMessage(), ex);
				}
				wrapper.setResponse(studyApprovalDto);
				wrapper.setSuccess(true);
				wrapper.setMessage(null);
			} else {
				wrapper.setResponse(studyApprovalDto);
				wrapper.setSuccess(false);
				wrapper.setMessage("Unable to update Study Approval");
			}
		}
		return wrapper;
	}

	/**
	 * Gets the study approvals by ID.
	 *
	 * @param id
	 *            The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getStudyApprovalsByID(int id) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getStudyApprovalsByID: {}", id);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		StudyApprovalDTO studyApprovalDto;
		StudyApproval studyApproval = null;
		try {
			studyApproval = studyApprovalService.getStudyApprovalsByID(id);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getStudyApprovalsByID.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (studyApproval != null) {
			studyApprovalDto = studyApprovalConverter.convert(studyApproval);
			wrapper.setResponse(studyApprovalDto);
		}
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the study approvals by UID.
	 *
	 * @param uid
	 *            The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getStudyApprovalsByUID(String uid) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getStudyApprovals by user: {}", uid);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApprovalDTO> studyApprovalDtos;
		List<StudyApproval> studyApprovals = null;
		try {
			studyApprovals = studyApprovalService.getStudyApprovalsByUID(uid);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getStudyApprovalsByUID.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		studyApprovalDtos = studyApprovalConverter.convertStudyApprovals(studyApprovals);
		wrapper.setResponse(studyApprovalDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the study approvals.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getStudyApprovals() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getStudyApprovals");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<StudyApprovalDTO> studyApprovalDtos;
		List<StudyApproval> studyApprovals = null;
		try {
			studyApprovals = studyApprovalService.getStudyApprovals();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying getStudyApprovals.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		studyApprovalDtos = studyApprovalConverter.convertStudyApprovals(studyApprovals);
		wrapper.setResponse(studyApprovalDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request data sources.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestDataSources() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getDataSources");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Source> listDatasources = new ArrayList<>();
		try {
			listDatasources = lookUpService.getDataSources();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying Request Data Sources.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		wrapper.setResponse(sourceConverter.convertSources(listDatasources));
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request types.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestTypes() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getRequestTypes");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestType> listRequestTypes = new ArrayList<>();
		try {
			listRequestTypes = lookUpService.getRequestTypes();
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while querying Request Types.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		wrapper.setResponse(requestTypeConverter.convertRequestTypes(listRequestTypes));
		return createResponseWrapper(wrapper);
	}

	/**
	 * Creates the response wrapper.
	 *
	 * @param wrapper
	 *            The wrapper.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@SuppressWarnings("rawtypes")
	private ResponseWrapper createResponseWrapper(ResponseWrapper wrapper) {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  creating Response Wrapper");
		}
		if (wrapper == null) {
			ResponseWrapper wrapperLocal = new ResponseWrapper();
			wrapperLocal.setMessage("Internal System Error. Could not perform requested Opeartion");
			wrapperLocal.setResponse(null);
			wrapperLocal.setSuccess(false);
			return wrapperLocal;
		}
		if (wrapper.getResponse() == null) {
			wrapper.setSuccess(true);
			wrapper.setResponse(null);
			wrapper.setMessage(CommonEnum.NO_RECORD_FOUND.getText());
			return wrapper;
		}
		Object object = wrapper.getResponse();
		String isOk;
		for (;;) {
			if (object instanceof ArrayList<?> || object instanceof StudyApprovalDTO || object instanceof CommentHistoryDTO || object instanceof RequestDTO) {
				isOk = CommonEnum.TRUE.getText();
			} else {
				isOk = CommonEnum.FALSE.getText();
			}
			break;
		}
		if (isOk.equalsIgnoreCase(CommonEnum.FALSE.getText())) {
			wrapper.setSuccess(true);
			wrapper.setResponse(null);
			wrapper.setMessage(CommonEnum.NO_RECORD_FOUND.getText());
			return wrapper;
		}
		if (!(object instanceof ArrayList<?>)) {
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
			wrapper.setResponse(object);
			return wrapper;
		}
		try {
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
			if (((ArrayList) object).size() == 1) {
				Object obj = ((ArrayList<?>) object).get(0);
				if (obj instanceof StudyApproval) {
					StudyApproval studyApproval = (StudyApproval) (obj);
					wrapper.setResponse(studyApproval);
				} else if (obj instanceof CommentHistory) {
					CommentHistory commentHistory = (CommentHistory) (obj);
					wrapper.setResponse(commentHistory);
				}
				return wrapper;
			}
		} catch (Exception ex) {
			LOGGER.error("Exception occurred in the method createResponseWrapper. Reason: " + ex.getMessage() + " ", ex);
		}
		return wrapper;
	}

	/**
	 * Get user details by email.
	 *
	 * @param email
	 *            The email.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getUserDetailsByEmail(String email) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  user details by email: {}", email);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		User user = null;
		try {
			user = userManagementService.getUserDetailsByEmail(email);
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occurred while querying getUserDetailsByEmail.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (user == null) {
			setNoRecordsFound(wrapper);
		} else {
			UserDTO userDto = userConverter.convert(user);
			wrapper.setSuccess(true);
			wrapper.setResponse(userDto);
		}
		return wrapper;
	}

	/**
	 * Get user details by id.
	 *
	 * @param id
	 *            The id.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getUserDetailsById(int id) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  user details by Id: {}", id);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		User user = null;
		try {
			user = userManagementService.getUserDetailsById(id);
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occurred while querying getUserDetailsById.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (user == null) {
			setNoRecordsFound(wrapper);
		} else {
			UserDTO userDto = userConverter.convert(user);
			wrapper.setSuccess(true);
			wrapper.setResponse(userDto);
		}
		return wrapper;
	}

	/**
	 * Gets the user role.
	 *
	 * @param uid
	 *            The uid.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getUserRole(int uid) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  user role by uid: {}", uid);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		String userRole = StringUtils.EMPTY;
		try {
			userRole = userManagementService.getUserRole(uid);
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occurred while querying user role.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		
		if (StringUtils.isBlank(userRole)) {
			wrapper.setMessage("No roles found.");
			wrapper.setResponse(null);
			wrapper.setSuccess(false);
		} else {
			wrapper.setSuccess(true);
			wrapper.setResponse(userRole);
		}
		return wrapper;
	}

	/**
	 * Gets the user details by username.
	 *
	 * @param username
	 *            The username.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getUserDetailsByUsername(String username) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getUserDetailsByUsername:" + username);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		User user = null;
		try {
			user = userManagementService.getUserDetailsByUsername(username);
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occured while querying user role.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (user == null) {
			wrapper.setMessage(ErrorEnum.USER_DOESNOT_EXIST.getErrorMessage());
			wrapper.setSuccess(false);
			wrapper.setResponse(null);
		} else {
			UserDTO userDto = userConverter.convert(user);
			wrapper.setSuccess(true);
			wrapper.setResponse(userDto);
		}
		return wrapper;
	}

	/**
	 * Gets the user counts by role and LDAP last refresh date/time
	 *
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getUserCountsAndLdapLastRefresh() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getUserCountsAndLdapLastRefresh");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<UserCount> userCounts;
		try {
			userCounts = userManagementService.getUserCountOnRole();
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occured while querying user count on role.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		Date lastRefreshed = null;
		try {
			lastRefreshed = userManagementService.getLatestUserRefreshHistory();
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occured while querying user count on role.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (null == userCounts || null == lastRefreshed) {
			setNoRecordsFound(wrapper);
		} else {
			wrapper.setSuccess(true);
			wrapper.setResponse(userCountConverter.populateUserCount(userCounts, lastRefreshed));
		}
		return wrapper;
	}

	/**
	 * Gets the user counts by role and LDAP last refresh date/time
	 *
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper refreshUserData(String username) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  refreshUserData");
		}
		
		UserRefreshHistory userRefreshHistory = userRefreshHistoryConverter.populateUserRefreshHistory(username);
		try {
			userManagementService.createUserRefreshHistory(userRefreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while creating userRefreshHistory", ex);
			throw new GenisisServiceException("Exception occured while creating userRefreshHistory", ex);
		}
		
		try {
			userManagementService.refreshUsers(username);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while pulling user data from LDAP", ex);
			userRefreshHistoryConverter.populateUserRefreshHistory(userRefreshHistory, ex.getMessage());
			userManagementService.updateUserRefreshHistory(userRefreshHistory);
			throw new GenisisServiceException("Exception occured while pulling user data from LDAP", ex);
		}
		
		try {
			userRefreshHistoryConverter.populateUserRefreshHistory(userRefreshHistory);
			userManagementService.updateUserRefreshHistory(userRefreshHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occured while updating userRefreshHistory", ex);
			// update job status as failed with error message details
			userRefreshHistoryConverter.populateUserRefreshHistory(userRefreshHistory, ex.getMessage());
			userManagementService.updateUserRefreshHistory(userRefreshHistory);
			throw new GenisisServiceException("Exception occured while updating userRefreshHistory", ex);
		}
		
		// update cache with latest user data
		ehcacheHelper.refreshUserData();
		
		return getUserCountsAndLdapLastRefresh();
	}

	/**
	 * Creates the request.
	 *
	 * @param requestIn
	 *            The requestIn.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper createRequest(RequestDTO requestIn) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: createRequest");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = requestConverter.convert(requestIn);
		User user = null;
		if (StringUtils.isBlank(request.getApproverId())) {
			try {
				user = this.userManagementService.getUserDetailsByUsername(request.getCreatedBy());
			} catch (Exception ex) {
				LOGGER.error("Exception occurred while getting User", ex);
			}
			if (user != null)
				request.setApproverId(userConverter.getUserApprover(user));
		}
		// Implement workflow
		boolean workflowProcessIdRequired = true;
		if (request.getId() != 0) {
			Request existingRequest = new Request();
			try {
				existingRequest = requestService.getRequestById(request.getId());
			} catch (Exception ex) {
				LOGGER.error("Exception occurred while getting existing Request", ex);
			}
			if (existingRequest.getId() != 0) {
				workflowProcessIdRequired = false;
			}
		}
		if (workflowProcessIdRequired) {
			String processInstanceId = StringUtils.EMPTY;
			try {
				processInstanceId = activitiService.startProcess(request.getCreatedBy());
			} catch (Exception e) {
				LOGGER.error("Unable to start process for " + request.getCreatedBy() + ". Reason: " + e.getMessage() + " ", e);
			}
			if (StringUtils.isBlank(processInstanceId)) {
				request.setProcessId(0);
			} else {
				int intProcessInstanceId = Integer.parseInt(processInstanceId);
				request.setProcessId(intProcessInstanceId);
			}
		}
		if (request.getProcessId() == 0) {
			wrapper.setResponse(requestConverter.convert(request));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request. Business process failed to start.");
			return wrapper;
		}
		int id = 0;
		try {
			id = requestService.createRequest(request);
		} catch (Exception e) {
			LOGGER.error("Unable to create Request." + e.getMessage() + " ", e);
		}
		if (id > 0) {
			this.preserveHistory(request);
			wrapper = this.getRequestByID(id);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
		} else {
			wrapper.setResponse(requestConverter.convert(request));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}

	/**
	 * Gets the request by ID.
	 *
	 * @param id
	 *            The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestByID(int id) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get Request by Id: {}", id);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		RequestDTO requestDto;
		Request request = new Request();
		try {
			request = requestService.getRequestById(id);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while querying getUserDetailsByEmail.", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		if (request != null) {
			requestDto = requestConverter.convert(request);
			wrapper.setResponse(requestDto);
		}
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the request tracking by ID.
	 *
	 * @param id
	 *            The id.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestTrackingByID(int id, String userRole) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get Audit Request by Id: {}", id);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		Request request = new Request();
		try {
			request = requestService.getRequestById(id);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting Requests by Id", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		if (request != null) {
			List<RequestHistory> trackingList = new ArrayList<>();
			try {
				trackingList = requestService.getRequestHistoriesWithCommentsById(id);
			} catch (Exception e) {
				LOGGER.error("Exception occurred while getting Request History", e);
				throw new GenisisServiceException(e.getMessage(), e);
			}
			if (trackingList == null || trackingList.isEmpty())
				wrapper.setResponse(null);
			else {
				// transform trackingList
				wrapper.setResponse(this.getAuditTrailWithComments(trackingList, userRole));
			}
		}
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by UID.
	 *
	 * @param uid
	 *            The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestsByUID(String uid) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get Request by User: {}", uid);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestDTO> requestDtos;
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getRequestsByUID(uid);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting Requests by UID", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requests = this.filterRequestList(requests, uid);
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by data managers.
	 *
	 * @param uid
	 *            The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestsByDataManagers(String uid) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get Request by Data Managers");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestDTO> requestDtos;
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getRequestsDataManagers(uid);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting Requests by Data Managers", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the requests by data source managers.
	 *
	 * @param uid
	 *            The uid.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getRequestsDataSourceManagers(String uid) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get Request by Data Source Managers");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestDTO> requestDtos;
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getRequestsDataSourceManagers(uid);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting Requests by Data Source Managers", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Filter request list.
	 *
	 * @param requests
	 *            The requests.
	 * @param uid
	 *            The uid.
	 * @return List<Request> This returns filteredRequestList.
	 */
	private List<Request> filterRequestList(List<Request> requests, String uid) {
		List<Request> filteredRequestList = new ArrayList<>();
		for (Request request : requests) {
			if (!(!StringUtils.equalsIgnoreCase(request.getCreatedBy(), uid) && StringUtils.equalsIgnoreCase(request.getStatusDescription(), "Draft"))) {
				filteredRequestList.add(request);
			}
		}
		return filteredRequestList;
	}

	/**
	 * Gets the all requests.
	 *
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getAllRequests() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get all Requests");
		}
		List<RequestDTO> requestDtos;
		ResponseWrapper wrapper = new ResponseWrapper();
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getAllRequests();
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting All Requests", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the all requests by study approval.
	 *
	 * @param studyApprovalId
	 *            The studyApprovalId.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getAllRequestsByStudyApproval(int studyApprovalId) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: getAllRequestsByStudyApproval: {}", studyApprovalId);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestDTO> requestDtos;
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getAllRequestsByStudyApproval(studyApprovalId);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting All Requests by Study Approval ", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Gets the all requests by status.
	 *
	 * @param status
	 *            The status.
	 * @return ResponseWrapper This returns ResponseWrapper.
	 */
	@Override
	public ResponseWrapper getAllRequestsByStaus(String status) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: get all Requests by Status: {}", status);
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<RequestDTO> requestDtos;
		List<Request> requests = new ArrayList<>();
		try {
			requests = requestService.getAllRequestsByStaus(status);
		} catch (Exception e) {
			LOGGER.error("Exception occurred while getting All Requests by Status ", e);
			throw new GenisisServiceException(e.getMessage(), e);
		}
		requestDtos = convertRequests(requests);
		wrapper.setResponse(requestDtos);
		return createResponseWrapper(wrapper);
	}

	/**
	 * Submit or modify.
	 *
	 * @param request
	 *            The request. status
	 */
	@Override
	public ResponseWrapper submitOrModify(RequestDTO request) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: Sumbit/Modify Request");
		}
		return this.createRequest(request);
	}

	/**
	 * This method is used to Persist an entitiy.
	 *
	 * @param id
	 *            The id.
	 * @param changeRequestIn
	 *            The changeRequestIn.
	 * @param operation
	 *            The operation.
	 * @return ResponseWrapper This returns wrapper.
	 * 
	 */
	@Override
	public ResponseWrapper persist(int id, RequestDTO changeRequestIn, String operation) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: Persisting: {}", operation);
		}
		
		ResponseWrapper wrapper = new ResponseWrapper();

		// check is operation allowed on a request
		if (!businessHelper.isOperationAllowed(operation)) {
			wrapper.setResponse(changeRequestIn);
			wrapper.setMessage(ErrorEnum.OPERATION_NOT_ALLOWED.getErrorMessage());
			wrapper.setSuccess(false);
			return wrapper;
		}
		
		Request changeRequest = new Request();
		if (StringUtils.equalsIgnoreCase(operation, "Update") || StringUtils.equalsIgnoreCase(operation, "Submitted")) {
			changeRequest = requestConverter.convert(changeRequestIn);
		}
		
		Request databaseRequest = new Request();
		try {
			databaseRequest = requestService.getRequestById(id);
		} catch (Exception ex) {
			LOGGER.error("Error occurred while getting Request by Id", ex);
		}
		
		if (databaseRequest == null) {
			changeRequest.setId(id);
			wrapper.setResponse(changeRequestIn);
			wrapper.setMessage("No such request found in database");
			wrapper.setSuccess(false);
			return wrapper;
		}
		
		String status = StringUtils.trim(databaseRequest.getStatusDescription());
		if (businessHelper.getWorkflowDeadEnds(status)) {
			wrapper.setResponse(changeRequestIn);
			wrapper.setMessage(ErrorEnum.OPERATION_NOT_ALLOWED.getErrorMessage());
			wrapper.setSuccess(false);
			return wrapper;
		}
		
		// Submitted, Returned, Sent, Request Accepted and Results Delivered
		if ((operation.equalsIgnoreCase(WorkflowStatusEnum.CANCELLED.getDesc()) && !businessHelper.isCancelAllowed(status)) 
				|| operation.equalsIgnoreCase(WorkflowStatusEnum.RETURNTODDM.getDesc()) && !businessHelper.isReturnToDDMAllowed(status)) {
			wrapper.setResponse(changeRequest);
			wrapper.setMessage(ErrorEnum.OPERATION_NOT_ALLOWED.getErrorMessage());
			wrapper.setSuccess(false);
			return wrapper;
		}
		
		requestConverter.populateRequest(id, changeRequestIn, operation, databaseRequest, changeRequest);
		
		boolean statusCompleteWorkflowProcess = this.completeWorkflowProcess(id, operation, changeRequestIn.getComments(), changeRequest);
		
		if (!statusCompleteWorkflowProcess) {
			wrapper.setResponse(requestConverter.convert(changeRequest));
			wrapper.setSuccess(false);
			wrapper.setMessage(operation + " failed. Error occurred on the workflow. Please contact system administrator. Id#" + id);
			return wrapper;
		}
		UUID uuid = UUID.randomUUID();
		changeRequest.setTaskId(uuid.toString());
		int submitOrModifyId = 0;
		try {
			submitOrModifyId = requestService.submitOrModify(changeRequest);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred on submitOrModify Request", ex);
		}
		if (submitOrModifyId > 0) {
			wrapper = this.getRequestByID(submitOrModifyId);
			wrapper.setSuccess(true);
			wrapper.setMessage(null);
			try {
				preserveHistory(requestService.getRequestById(submitOrModifyId));
			} catch (Exception ex) {
				LOGGER.error(CommonEnum.EXCEPTION_QUERYING_REQUESTS.getText(), ex);
			}
			if (!StringUtils.isBlank(changeRequestIn.getComments())) {
				// Save Comments for declines in CommentHistory
				CommentHistory commentHistory = new CommentHistory();
				commentHistory.setComments(changeRequestIn.getComments());
				Request request = new Request();
				try {
					request = requestService.getRequestById(submitOrModifyId);
				} catch (Exception ex) {
					LOGGER.error(CommonEnum.EXCEPTION_QUERYING_REQUESTS.getText(), ex);
				}
				int statusId = lookUpService.getStatusId(request.getStatusDescription());
				WorkflowStatus workflowStatus = new WorkflowStatus();
				workflowStatus.setId(statusId);
				commentHistory.setWorkflowStatus(workflowStatus);
				commentHistory.setRequest(request);
				commentHistory.setCreatedBy(changeRequest.getModifiedBy());
				commentHistory.setType("action");
				commentHistory.setTaskId(uuid.toString());
				// All Action comments are treated as General comments
				if (commentHistory.getCommentTypeId() == 0) {
					commentHistory.setCommentTypeId(2);
					CommentType commentType = lookUpService.getCommentType(commentHistory.getCommentTypeId());
					commentHistory.setCommentType(commentType);
				}
				commentHistoryService.submitOrModify(commentHistory);
			}
		} else {
			wrapper.setResponse(requestConverter.convert(changeRequest));
			wrapper.setSuccess(false);
			wrapper.setMessage("Unable to create Request");
		}
		return wrapper;
	}

	/**
	 * Complete workflow process.
	 *
	 * @param requestId
	 *            The requestId.
	 * @param operation
	 *            The operation.
	 * @param changeRequest
	 *            The changeRequest.
	 * @return boolean This returns true, if successful.
	 */
	private boolean completeWorkflowProcess(int requestId, String operation, String commentsIn, Request changeRequest) {
		boolean success = true;
		String requestorId = null;
		String approverId = null;
		String userId = null;
		String comments = (commentsIn == null) ? StringUtils.EMPTY : commentsIn;
		Request request = new Request();
		
		try {
			request = requestService.getRequestById(requestId);
		} catch (Exception ex) {
			LOGGER.error(CommonEnum.EXCEPTION_QUERYING_REQUESTS.getText(), ex);
		}
		
		RequestType requestType = null;
		Source source = null;
		if (request != null) {
			requestType = request.getRequestType();
			source = request.getSource();
		}
		
		User requesterUser = null;
		User approverUser = null;
			
		try {
			requesterUser = userManagementService.getUserDetailsByUsername(request.getCreatedBy());
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while getting Requesters", ex);
		}
		try {
			approverUser = userManagementService.getUserDetailsByUsername(changeRequest.getModifiedBy());
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while getting approvers", ex);
		}
	
		
		String status = request.getStatusDescription();
		request.getStudyApproval();
		StudyApproval studyApproval = null;
		
		if (request != null) {
			studyApproval = request.getStudyApproval();
		}

		if (approverUser == null) {
			approverId = StringUtils.EMPTY;
		} else {
			if (!StringUtils.isBlank(approverUser.getUsername()))
				approverId = approverUser.getUsername();
		}
		
		if (requesterUser == null) {
			requestorId = StringUtils.EMPTY;
		} else {
			if (!StringUtils.isBlank(requesterUser.getUsername()))
				requestorId = requesterUser.getUsername();
		}
		
		JSONObject json = new JSONObject();
		json.put(CommonEnum.EMAIL_OPERATION.getText(), operation);
		json.put(CommonEnum.EMAIL_REQUESTOR_ID.getText(), requestorId);
		json.put(CommonEnum.EMAIL_APPROVER_DECISION.getText(), status);
		json.put(CommonEnum.EMAIL_FULFILLMENT_DECISION.getText(), "");
		json.put(CommonEnum.EMAIL_RESULT_ACCEPTANCE_DECISION.getText(), "");
		json.put(CommonEnum.EMAIL_APPROVER_ID.getText(), approverId);
		json.put(CommonEnum.EMAIL_CC_USERS.getText(), "");
		json.put(CommonEnum.EMAIL_RESULTS_DELIVER_DECISION.getText(), "");
		String pid = String.valueOf(request.getProcessId());
		
		List<User> admins = null;
		List<User> ddms = null;
		List<User> dsms = null;
		try {
			admins = userManagementService.getUserDetailsByRoleTypeId(RoleTypeEnum.ADMIN.getId());
			ddms = userManagementService.getUserDetailsByRoleTypeId(RoleTypeEnum.DDM.getId());
			dsms = userManagementService.getUserDetailsByRoleTypeId(RoleTypeEnum.DSM.getId());
		} catch (GenisisServiceException ex) {
			LOGGER.error(ErrorEnum.ERR_GETUSERS_BY_ROLEID.getErrorMessage(), ex);
		}
		
		userId = request.getModifiedBy();
		if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.SUBMITTED.getDesc())) {
			emailHelper.populateEmailDetailsForSubmit(json, requestId, request, requesterUser, requestorId, admins, ddms);
			userId = request.getCreatedBy();
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.RETURNED.getDesc())) {
			emailHelper.populateEmailDetailsForReturned(json, requestId, request, comments, requesterUser);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.DENIED.getDesc())) {
			emailHelper.populateEmailDetailsForDenied(json, requestId, request, comments, requesterUser);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.SENT.getDesc())) {
			emailHelper.populateEmailDetailsForSent(json, requestId, request, commentsIn, requestType, source, studyApproval, requesterUser, admins, dsms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.REQUESTNOTACCEPTED.getDesc())) {
			emailHelper.populateEmailDetailsForNotAccepted(json, requestId, request, comments, requesterUser, admins, ddms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.REQUESTACCEPTED.getDesc())) {
			emailHelper.populateEmailDetailsForAccepted(json, requestId, request, comments, requesterUser);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.RESULTSDELIVERED.getDesc())) {
			emailHelper.populateEmailDetailsForResultsDelivered(json, requestId, request, comments, requesterUser, admins, ddms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.RESULTSNOTACCEPTED.getDesc())) {
			emailHelper.populateEmailDetailsForResultsNotAccepted(json, requestId, request, comments, requesterUser, admins, dsms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.RESULTSACCEPTED.getDesc())) {
			emailHelper.populateEmailDetailsForResultsAccepted(json, requestId, request, comments, requesterUser, admins, dsms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.CANCELLED.getDesc())) {
			emailHelper.populateEmailDetailsForCancelledRequest(json, requestId, request, requesterUser, admins, ddms, dsms);
		} else if (StringUtils.equalsIgnoreCase(operation, WorkflowStatusEnum.RETURNTODDM.getDesc())) {
			emailHelper.populateEmailDetailsForReturnToDDMRequest(json, requestId, request, comments, requesterUser, admins, ddms);
		}
		
		if (businessHelper.isNotificationAllowed(operation)) {
			try {
				activitiService.claimAndCompleteHumanTask(pid, userId, json);
			} catch (Exception e) {
				LOGGER.error("Unable to Claim and Complete Task for " + operation + " . Reason: " + e.getMessage() + "  ", e);
				success = false;
			}
		}
		
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("CompleteWorkflowProcess: Success status: {}", success);
		}
		
		return success;
	}

	/**
	 * Preserve history.
	 *
	 * @param request
	 *            The request.
	 */
	private void preserveHistory(Request request) {
		RequestHistory requestHistory = requestHistoryConverter.populateRequestHistory(request);
		int returnId = 0;
		try {
			returnId = requestService.submitOrModify(requestHistory);
		} catch (Exception ex) {
			LOGGER.error("Exception occurred while submitOrModify Request ", ex);
		}
		if (returnId > 0) {
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info(requestHistory.getRequest().getId() + " with status " + requestHistory.getStatusDescription() + " added to the history successfully!!");
			}
		} else {
			LOGGER.warn(requestHistory.getRequest().getId() + " with status " + requestHistory.getStatusDescription() + " could not be added to the history.");
		}
	}

	/**
	 * Creates the comment history.
	 *
	 * @param commentHistoryDto
	 *            The commentHistory.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper createCommentHistory(CommentHistoryDTO commentHistoryDto) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: createCommentHistory");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		CommentHistory commentHistory;
		if (null == commentHistoryDto) {
			LOGGER.error("comment history is null");
			throw new GenisisServiceException("comment history is null");
		} else {
			commentHistory = commentHistoryConverter.convert(commentHistoryDto);
			int statusId = lookUpService.getStatusId(commentHistory.getStatus());
			Request request = requestService.getRequestById(commentHistory.getRequestId());
			CommentType commentType = lookUpService.getCommentType(commentHistory.getCommentTypeId());
			if (commentType == null || request == null || statusId == 0) {
				wrapper.setResponse(commentHistoryDto);
				wrapper.setSuccess(false);
				wrapper.setMessage("Unable to create Comment History. One of the parameter (Comment Type/ Request Id / Status) is incorrect.");
				return wrapper;
			}

			WorkflowStatus workflowStatus = new WorkflowStatus();
			workflowStatus.setId(statusId);
			commentHistory.setWorkflowStatus(workflowStatus);
			commentHistory.setRequest(request);
			commentHistory.setTaskId(request.getTaskId());
			commentHistory.setCommentType(commentType);
			int id = commentHistoryService.submitOrModify(commentHistory);
			if (id > 0) {
				commentHistoryDto.setRequestId(commentHistory.getRequestId());
				commentHistoryDto.setStatus(commentHistory.getStatus());
				wrapper.setResponse(commentHistoryDto);
				wrapper.setSuccess(true);
				wrapper.setMessage(null);
			} else {
				wrapper.setResponse(commentHistoryDto);
				wrapper.setSuccess(false);
				wrapper.setMessage("Unable to create Comment History");
			}
		}
		return wrapper;
	}

	/**
	 * Gets the audit trail with comments.
	 *
	 * @param trackingList
	 *            The trackingList.
	 * @return List<RequestHistoryDTO> This returns returnRequestHistory.
	 */
	private List<RequestHistoryDTO> getAuditTrailWithComments(List<RequestHistory> trackingList, String userRole) {
		List<RequestHistoryDTO> returnRequestHistories = new ArrayList<>();
		RequestHistory requestHistories = null;
		List<CommentHistory> commentHistories = new ArrayList<>();
		String pTaskId = StringUtils.EMPTY;
		// Get a List of Request Histories
		for (Object object : trackingList) {
			Object[] obj = (Object[]) object;
			RequestHistory requestHistory = (RequestHistory) obj[0];
			CommentHistory commentHistory = (CommentHistory) obj[1];
			if (requestHistory.getTaskId() != pTaskId) {
				if (requestHistories != null) {
					RequestHistoryDTO element = requestHistoryConverter.convert(requestHistories, commentHistories);
					returnRequestHistories.add(element);
				}
				pTaskId = requestHistory.getTaskId();
				commentHistories = new ArrayList<>();
				requestHistories = requestHistory;
			}
			if (commentHistory != null && !(StringUtils.equals(userRole, CommonEnum.REQUESTER_ROLE.getText()) && commentHistory.getCommentType().getId() == 1))
				commentHistories.add(commentHistory);
		}
		if (requestHistories != null) {
			RequestHistoryDTO element = requestHistoryConverter.convert(requestHistories, commentHistories);
			returnRequestHistories.add(element);
		}
		return returnRequestHistories;
	}

	/**
	 * Perform table copy transfer.
	 *
	 * @param jsonRequest
	 *            The jsonRequest.
	 * @param requestId
	 *            The requestId.
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper performTableCopyTransfer(CopyTableSource jsonRequest, int requestId) throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  performTableCopyTransfer");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		wrapper.setSuccess(true);
		String[] command = new String[16];
		businessHelper.validateCopyTableSource(copyTableSource, jsonRequest, wrapper);
		if (wrapper.getSuccess()) {
			command = (String[]) wrapper.getResponse();
		}
		Process process = null;
		BufferedReader reader = null;
		String line = StringUtils.EMPTY;
		if (LOGGER.isInfoEnabled()) {
			String strCommand = Arrays.toString(command);
			LOGGER.info("Table/Copy arguments: {}", strCommand);
		}
		try {
			Runtime runtime = Runtime.getRuntime();
			process = runtime.exec(command);
			process.waitFor();
			reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
			while ((line = reader.readLine()) != null) {
				wrapper.setResponse(line);
			}
			if (LOGGER.isInfoEnabled()) {
				LOGGER.info("DataOps  Response: " + wrapper.getResponse());
			}
		} catch (Exception ex) {
			LOGGER.error("Error occurred while copying table(s). Reason: " + ex.getMessage() + " ", ex);
			wrapper.setResponse(jsonRequest);
			wrapper.setMessage("Copy Table failed. Please contact system administrator.");
			wrapper.setSuccess(false);
		} finally {
			try {
				if (process != null && process.isAlive())
					process.destroy();
				if (reader != null)
					reader.close();
			} catch (IOException ex) {
				LOGGER.error("Error occurred while closing the resources. Reason: " + ex.getMessage(), ex);
			}
		}
		return wrapper;
	}

	/**
	 * Gets the copy table domains.
	 *
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getCopyTableDomainDetails() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService: Copy Table Domains");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		wrapper.setMessage("");
		wrapper.setSuccess(true);
		wrapper.setResponse(copyTableDomains);
		return wrapper;
	}
	
	/**
	 * Gets the Request Status  counts 
	 *
	 * @return ResponseWrapper This returns wrapper.
	 */
	@Override
	public ResponseWrapper getRequestStatusCounts() throws GenisisServiceException {
		if (LOGGER.isInfoEnabled()) {
			LOGGER.info("BusinessService:  getRequestStatusCounts");
		}
		ResponseWrapper wrapper = new ResponseWrapper();
		List<WorkflowStatus> requestStatusCounts = null;
		try {
			requestStatusCounts = requestService.getRequestStatusCounts();
		} catch (Exception ex) {
			LOGGER.error("BusinessService: Exception occured while querying Request Status Counts.", ex);
			throw new GenisisServiceException(ex.getMessage(), ex);
		}
		if (null == requestStatusCounts) {
			setNoRecordsFound(wrapper);
		} else {
			wrapper.setSuccess(true);
			wrapper.setResponse(workflowStatusConverter.convert(requestStatusCounts));
		}
		return wrapper;
	}

	/**
	 * This method is used to prepare list of request response payload.
	 * 
	 * @param requests
	 *            The requests
	 * @return the list of request payload.
	 */
	private List<RequestDTO> convertRequests(List<Request> requests) {
		List<RequestDTO> requestDtos;
		if (requests == null || requests.isEmpty()) {
			return Collections.emptyList();
		} else {
			requestDtos = requestConverter.convertRequests(requests);
		}
		return requestDtos;
	}

	private void setNoRecordsFound(ResponseWrapper wrapper) {
		wrapper.setMessage(ErrorEnum.ERR_NO_RECORDS_FOUND.getErrorMessage());
		wrapper.setSuccess(false);
		wrapper.setResponse(null);
	}

	/**
	 * @return the copyTableSource
	 */
	public CopyTableSource getCopyTableSource() {
		return copyTableSource;
	}

	/**
	 * @param copyTableSource
	 *            the copyTableSource to set
	 */
	public void setCopyTableSource(CopyTableSource copyTableSource) {
		this.copyTableSource = copyTableSource;
	}

	/**
	 * @param copyTableDomains
	 *            the copyTableDomains to set
	 */
	@Autowired
	public void setCopyTableDomains(CopyTableDomains copyTableDomains) {
		this.copyTableDomains = copyTableDomains;
	}

	/**
	 * Gets the copy table source.
	 *
	 * @param copyTableSource
	 *            the copy table source
	 *
	 */
	@Autowired
	public void getCopyTableSource(CopyTableSource copyTableSource) {
		this.copyTableSource = copyTableSource;
	}

	/**
	 * Sets the look up service.
	 *
	 * @param lookUpService
	 *            The lookUpService.
	 */
	@Autowired
	public void setLookUpService(LookUpService lookUpService) {
		this.lookUpService = lookUpService;
	}

	/**
	 * Sets the activiti service.
	 *
	 * @param activitiService
	 *            The activitiService.
	 */
	@Autowired
	public void setActivitiService(ActivitiService activitiService) {
		this.activitiService = activitiService;
	}

	/**
	 * Sets the request service.
	 *
	 * @param requestService
	 *            The requestService.
	 */
	@Autowired
	public void setRequestService(RequestService requestService) {
		this.requestService = requestService;
	}

	/**
	 * Sets the study approval service.
	 *
	 * @param studyApprovalService
	 *            The studyApprovalService.
	 */
	@Autowired
	public void setStudyApprovalService(StudyApprovalService studyApprovalService) {
		this.studyApprovalService = studyApprovalService;
	}

	/**
	 * Sets the comment history service.
	 *
	 * @param commentHistoryService
	 *            The commentHistoryService.
	 */
	@Autowired
	public void setCommentHistoryService(CommentHistoryService commentHistoryService) {
		this.commentHistoryService = commentHistoryService;
	}

	/**
	 * Set the user management service.
	 *
	 * @param userManagementService
	 *            The userManagementService.
	 */
	@Autowired
	public void setUserManagementService(UserManagementService userManagementService) {
		this.userManagementService = userManagementService;
	}

	/**
	 * @param genisisPropertiesUtil
	 *            the genisisPropertiesUtil to set
	 */
	public void setGenisisPropertiesUtil(GenisisPropertiesUtil genisisPropertiesUtil) {
		this.genisisPropertiesUtil = genisisPropertiesUtil;
	}

	/**
	 * @param requestConverter
	 *            the requestConverter to set
	 */
	public void setRequestConverter(RequestConverter requestConverter) {
		this.requestConverter = requestConverter;
	}

	/**
	 * @param userConverter
	 *            the userConverter to set
	 */
	public void setUserConverter(UserConverter userConverter) {
		this.userConverter = userConverter;
	}

	/**
	 * @param studyApprovalConverter
	 *            the studyApprovalConverter to set
	 */
	public void setStudyApprovalConverter(StudyApprovalConverter studyApprovalConverter) {
		this.studyApprovalConverter = studyApprovalConverter;
	}

	/**
	 * @param commentHistoryConverter
	 *            the commentHistoryConverter to set
	 */
	public void setCommentHistoryConverter(CommentHistoryConverter commentHistoryConverter) {
		this.commentHistoryConverter = commentHistoryConverter;
	}

	/**
	 * @param sourceConverter
	 *            the sourceConverter to set
	 */
	public void setSourceConverter(SourceConverter sourceConverter) {
		this.sourceConverter = sourceConverter;
	}

	/**
	 * @param requestTypeConverter
	 *            the requestTypeConverter to set
	 */
	public void setRequestTypeConverter(RequestTypeConverter requestTypeConverter) {
		this.requestTypeConverter = requestTypeConverter;
	}

	/**
	 * @param requestHistoryConverter
	 *            the requestHistoryConverter to set
	 */
	public void setRequestHistoryConverter(RequestHistoryConverter requestHistoryConverter) {
		this.requestHistoryConverter = requestHistoryConverter;
	}

	/**
	 * @param userCountConverter
	 *            the userCountConverter to set
	 */
	public void setUserCountConverter(UserCountConverter userCountConverter) {
		this.userCountConverter = userCountConverter;
	}

	/**
	 * @param userRefreshHistoryConverter
	 *            the userRefreshHistoryConverter to set
	 */
	public void setUserRefreshHistoryConverter(UserRefreshHistoryConverter userRefreshHistoryConverter) {
		this.userRefreshHistoryConverter = userRefreshHistoryConverter;
	}

	/**
	 * @param businessHelper
	 *            the businessHelper to set
	 */
	public void setBusinessHelper(BusinessHelper businessHelper) {
		this.businessHelper = businessHelper;
	}

	/**
	 * 
	 * @param emailHelper
	 *            the emailHelper to set
	 */
	public void setEmailHelper(EmailHelper emailHelper) {
		this.emailHelper = emailHelper;
	}
	
	/**
	 * @param ehcacheHelper the ehcacheHelper to set
	 */
	public void setEhcacheHelper(EhcacheHelper ehcacheHelper) {
		this.ehcacheHelper = ehcacheHelper;
	}
}
